﻿using System;
using System.Collections.Generic;
using System.Net;
using System.Reflection;
using System.Threading;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Server;
using gov.va.med.VBECS.VistaLinkServer.Core;
using gov.va.med.vbecs.Common.AppServices;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages;
using IServer = gov.va.med.vbecs.Common.IServer;

namespace gov.va.med.VBECS.VistaLinkServer
{
    /// <summary>
    /// Implements Vista Link server functionality
    /// </summary>
    internal class Server : IServer
    {
        public event EventHandler<ThreadExceptionEventArgs> FatalErrorOccured;
        private static readonly VistALinkMessageXmlFactory MessageFactory = new VistALinkMessageXmlFactory(new RpcParameterXmlFactory());
        private static readonly VistALinkRpcManager RpcManager = new VistALinkRpcManager();

        public List<PublishedProcedure> RpcList;

        // Inner TCP/IP server
        private Communication.Server.IServer _server;
        // Logger
        private readonly ILogger _logger =
            LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);
        // Events Logger
        private readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

        public void Start()
        {
            if (_server != null) return; // The server is already started

            // Read port number from configuration
            var portNumber =
                (int) GlobalContext.Instance().AppSettingsReader.GetValue("ListenerPortNumber", typeof (int));
            _server = ServerFactory.CreateServer(new IPEndPoint(IPAddress.Any, portNumber));

            _server.ClientDisconnected += server_client_disconnected;
            _server.ClientConnected += server_client_connected;
            _server.Start();
            _logger.Debug(string.Format("A server started (on port {0})", portNumber));
            _eventsLogger.Debug(string.Format("A server started (on port {0})", portNumber));
        }

        public void Stop()
        {
            if (_server == null) return; // The server is not started
            _server.Stop();
            _server = null;
        }

        private void server_client_connected(object sender, ClientDummyEventArgs e)
        {
            _logger.Debug("A new client is connected. Client Id = " + e.Client.Id);

            //Register to MessageReceived event to receive messages from new client
            e.Client.MessageReceived += client_message_received;
        }

        private void server_client_disconnected(object sender, ClientDummyEventArgs e)
        {
            _logger.Debug("A client is disconnected! Client Id = " + e.Client.Id);
        }

        private void client_message_received(object sender, MessageEventArgs e)
        {
            //Server only accepts RawMessages
            if (!(e.Message is RawDataMessage))
            {
                _logger.Warn(string.Format("Client sent unknown message: [type={0}]", (e.Message == null ? "null" : e.Message.GetType().ToString())));
                return;
            }
            //construct text message
            var message = new TextMessage(((RawDataMessage)e.Message).Data);

            //Get a reference to the client
            var client = (IClientDummy) sender;
            _logger.Debug("Client sent a message: " + new TextMessage(message.Text).Text +
                          " (Client Id = " + client.Id + ")");

            VistALinkMessage response;
            try
            {
                var incomingMessage = MessageFactory.ParseXmlCreateVistALinkMessage(message.Text);
                if (incomingMessage == null)
                    throw (new VistALinkException("Unknown message was received from remote party. It cannot be identified as a VistALink message."));

                //Note: It looks like we don't handle ping request messages, since they are not derived from RpcRequestMessage.
                if (!(incomingMessage is RpcRequestMessage))
                    throw (new VistALinkException(String.Format("Message of {0} type was received from client while message of {1} type was expected.",
                        incomingMessage.GetType().Name, typeof(RpcRequestMessage).Name)));

                try
                {
                    response = new RpcResponseMessage(RpcManager.ExecuteRpcRequest(((RpcRequestMessage)incomingMessage).RpcRequest, IPAddress.Parse("127.0.0.1"), RpcList));
                }
                catch (Exception xcp)
                {
                    //Log exception but continue listening for other messages.
                    _logger.Error(string.Format("Exception occurred while executing RPC. Message:\r\n{0}", message.Text), xcp);
                    response = new RpcFaultMessage(new FaultInfo(FaultCode.Server, "Exception occurred while executing RPC", "", "", xcp.GetType().Name, xcp.Message));
                }
            }
            catch (Exception xcp)
            {
                //Log exception but continue listening for other messages.
                _logger.Error(string.Format("Exception occurred while processing client request. Message:\r\n{0}", message.Text), xcp);
                _eventsLogger.Error(string.Format("Exception occurred while processing client request. Message:\r\n{0}", message.Text), xcp);
                response = new RpcFaultMessage(new FaultInfo(FaultCode.Server, "Exception occurred while processing client request", "", "", xcp.GetType().Name, xcp.Message));
            }

            //Send reply message to the client
            client.Send( new RawDataMessage(response.GetBytes()) );
        }
    }
}
